import asyncio
import websockets
from websockets import exceptions as ws_exceptions

connected = set()

async def handle_websocket(websocket, path):
    connected.add(websocket)
    try:
        await websocket.wait_closed()
    finally:
        connected.remove(websocket)

async def capture_and_send_frames():
    ffmpeg_cmd = [
        'ffmpeg',
        '-rtsp_transport', 'tcp',
        '-i', 'rtsp://127.0.0.1:8554/stream0',
        '-c:v', 'mjpeg',
        '-q:v', '2',
        '-f', 'mjpeg',
        'pipe:1'
    ]
    
    process = await asyncio.create_subprocess_exec(
        *ffmpeg_cmd,
        stdout=asyncio.subprocess.PIPE,
        stderr=asyncio.subprocess.DEVNULL
    )

    buffer = b''
    try:
        while True:
            data = await process.stdout.read(4096)
            if not data:
                break
            
            buffer += data
            while True:
                start = buffer.find(b'\xff\xd8')
                if start == -1:
                    break
                
                end = buffer.find(b'\xff\xd9', start)
                if end == -1:
                    break
                
                end += 2
                frame = buffer[start:end]
                buffer = buffer[end:]

                # Broadcast to all connected clients
                for ws in connected.copy():
                    try:
                        await ws.send(frame)
                    except ws_exceptions.ConnectionClosed:
                        connected.discard(ws)
    finally:
        process.terminate()
        await process.wait()

async def main():
    async with websockets.serve(handle_websocket, '0.0.0.0', 8765):
        await capture_and_send_frames()

if __name__ == '__main__':
    try:
        asyncio.run(main())
    except KeyboardInterrupt:
        print("Server stopped.")